#! /bin/sh

SERVER_ROOT="/opt/TrendMicro/Pixiebob/Server/"
UI_ROOT="/opt/TrendMicro/Pixiebob/UI/"
SERVER_DATA_ROOT="${SERVER_ROOT}init/data/"
SERVER_CONF_ROOT="${SERVER_ROOT}init/config/"
SERVER_CRON_ROOT="${SERVER_ROOT}init/cronjob/"
SERVER_API_DATA_ROOT="${SERVER_ROOT}api/data/"
AGENT_ROOT="${SERVER_ROOT}init/agent/"
INIT_PIXIEBOB_LOG="${SERVER_ROOT}logs/init_pixiebob.log"
INIT_PIXIEBOB_FLAG="${SERVER_ROOT}init/pixiebob_initialized.flag"
MIGRATE_FLAG="${SERVER_ROOT}init/migrate.flag"
OPENVA_UPGRADE_FLAG="${SERVER_ROOT}init/from_openva_upgrade.flag"
SERVER_PATCH_ROOT="${SERVER_ROOT}init/patch/"

SERVER_MODE="$(cat /opt/TrendMicro/Pixiebob/server_mode)"
RPM_VERSION="$(cat /opt/TrendMicro/Pixiebob/Server/rpm_version)"

TEXTUI_ACCOUNTS="/tmp/textUI_accounts"

DUMP_CONFIG="|/usr/bin/sh /opt/TrendMicro/Pixiebob/common_modules/common/dump_handler.sh %p %e %E 0"
DUMP_CONFIG_ESC="|\/usr\/bin\/sh \/opt\/TrendMicro\/Pixiebob\/common_modules\/common\/dump_handler.sh %p %e %E 0"
DUMP_PATTERN="kernel.core_pattern"

############################################################
##               Reload nginx config                      ##
############################################################
reload_nginx_conf()
{
    echo "[`date`] Reload nginx configuration" >> ${INIT_PIXIEBOB_LOG} 2>&1

    conf_file=""
    case "${SERVER_MODE}" in
      "ALL")
          conf_file="default_all.conf"
          ;;
      "UC")
          conf_file="default_uc.conf"
          ;;
      "UCRR")
          conf_file="default_ucrr.conf"
          ;;
      "UCLR")
          conf_file="default_uclr.conf"
    esac

    if [ "${conf_file}" != "" ]; then
        rm -f "/etc/nginx/conf.d/${conf_file}"
        cp -f "/opt/TrendMicro/Pixiebob/Server/init/config/nginx/${conf_file}" /etc/nginx/conf.d/

        MGMT_V4_ADDR=`ip -f inet -o addr show eth0 | sed -n '1p' | awk '{print $4}' | sed 's/\/.*$//g'`
        if [ "$MGMT_V4_ADDR" != "" ]; then
            MGMT_V4_LISTEN_STR="listen ${MGMT_V4_ADDR}:443;"
        else
            MGMT_V4_LISTEN_STR=""
        fi

        MGMT_V6_ADDR=`ip -f inet6 -o addr show eth0 | grep -v tentative |  grep -v 'local' | grep -v 'dynamic' | sed -n '1p' | awk '{print $4}' | sed 's/\/.*$//g' | tr '[a-z]' '[A-Z]'`
        local mgmt_upcase=$(echo ${MGMT_V6_ADDR:0:5} | tr '[a-z]' '[A-Z]')

        if [ "$MGMT_V6_ADDR" != "" ] && [ "${mgmt_upcase}" != "FE80:" ] ; then
            MGMT_V6_LISTEN_STR="listen [${MGMT_V6_ADDR}]:443;"
        else
            MGMT_V6_LISTEN_STR=""
        fi

        sed -i "s/\[MGMT_IPV4\]/${MGMT_V4_LISTEN_STR}/g" "/etc/nginx/conf.d/${conf_file}"
        sed -i "s/\[MGMT_IPV6\]/${MGMT_V6_LISTEN_STR}/g" "/etc/nginx/conf.d/${conf_file}"

        echo "[`date`] Reload nginx" >> ${INIT_PIXIEBOB_LOG} 2>&1
        systemctl restart nginx.service
    fi
}

############################################################
##               Initialize Database                      ##
############################################################
init_database() {
    echo "[`date`] Init database." >> ${INIT_PIXIEBOB_LOG} 2>&1
    # Setup environment
    /usr/pgsql-9.6/bin/postgresql96-setup initdb  >> ${INIT_PIXIEBOB_LOG} 2>&1

    # Change local setting to trust
    /usr/bin/cp -f ${SERVER_CONF_ROOT}pgsql/pg_hba.conf.sample /var/lib/pgsql/9.6/data/pg_hba.conf >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/cp -f ${SERVER_CONF_ROOT}pgsql/postgresql.conf.sample /var/lib/pgsql/9.6/data/postgresql.conf >> ${INIT_PIXIEBOB_LOG} 2>&1

    # Restart postgresql service
    systemctl enable postgresql-9.6.service  >> ${INIT_PIXIEBOB_LOG} 2>&1
    systemctl restart postgresql-9.6.service  >> ${INIT_PIXIEBOB_LOG} 2>&1

    # Create user and database
    /usr/pgsql-9.6/bin/createuser -h 127.0.0.1 -U postgres pixiebob >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/psql -h 127.0.0.1 -U postgres -c "ALTER USER pixiebob CREATEDB;" >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/pgsql-9.6/bin/createdb pixiebob -h 127.0.0.1 -U pixiebob >> ${INIT_PIXIEBOB_LOG} 2>&1

    /usr/pgsql-9.6/bin/createuser -h 127.0.0.1 -U postgres taxiiserver >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/psql -h 127.0.0.1 -U postgres -c "ALTER USER taxiiserver CREATEDB;" >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/pgsql-9.6/bin/createdb taxiiserver -h 127.0.0.1 -U taxiiserver >> ${INIT_PIXIEBOB_LOG} 2>&1

    /usr/bin/psql -h 127.0.0.1 -U pixiebob -d pixiebob -c "CREATE TABLE alembic_version (version_num varchar(32));" >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/psql -h 127.0.0.1 -U taxiiserver -d taxiiserver -c "CREATE TABLE alembic_version (version_num varchar(32));" >> ${INIT_PIXIEBOB_LOG} 2>&1

}

init_data() {
    echo "[`date`] Initialize data" >> ${INIT_PIXIEBOB_LOG} 2>&1
    # Execute init scripts
    systemctl start postgresql-9.6.service >> ${INIT_PIXIEBOB_LOG} 2>&1

    cd ${SERVER_DATA_ROOT}
    export DATABASE_URL="postgresql://pixiebob@/pixiebob?application_name=DDD-portal-installer"
    export DATABASE_USER="pixiebob"
    export DATABASE_NAME="pixiebob"
    export APP_SETTINGS="config.DevelopmentConfig"
    export PYTHONPATH="/opt/TrendMicro/Pixiebob/Server/api:/opt/TrendMicro/Pixiebob/common_modules"

    python manage.pyc db upgrade >> ${INIT_PIXIEBOB_LOG} 2>&1
    python manage.pyc init >> ${INIT_PIXIEBOB_LOG} 2>&1
    python manage.pyc sp -f "${SERVER_DATA_ROOT}resource/init_files.txt" >> ${INIT_PIXIEBOB_LOG} 2>&1

    # Set server mode in index.html (/opt/TrendMicro/Pixiebob/UI/app/index.html)
    sed -i "s/\[SERVER_MODE\]/'${SERVER_MODE}'/g" "${UI_ROOT}app/index.html"
    sed -i "s/\[SERVER_MODE\]/'${SERVER_MODE}'/g" "${UI_ROOT}app/log_detail.html"
    sed -i "s/\[SERVER_MODE\]/'${SERVER_MODE}'/g" "${UI_ROOT}app/host_detail.html"
    sed -i "s/\[SERVER_MODE\]/'${SERVER_MODE}'/g" "${UI_ROOT}app/dddna.html"

    /usr/bin/opentaxii-upgrade-db >> ${INIT_PIXIEBOB_LOG} 2>&1
}

init_log_data() {
    echo "[`date`] Initialize log data" >> ${INIT_PIXIEBOB_LOG} 2>&1
    mkdir -p /var/log_data/download_package/tmp >> ${INIT_PIXIEBOB_LOG} 2>&1
    mkdir -p /var/log_data/download_package/local_cache >> ${INIT_PIXIEBOB_LOG} 2>&1

    if ! psql -U pixiebob -lqt | cut -d \| -f 1 | grep -qw pixiebob_log; then
        echo "[`date`] Log database does not exist, create it." >> ${INIT_PIXIEBOB_LOG} 2>&1
        mkdir -p /var/log_data/tablespace >> ${INIT_PIXIEBOB_LOG} 2>&1
        chown postgres:postgres /var/log_data/tablespace >> ${INIT_PIXIEBOB_LOG} 2>&1

        /usr/bin/psql -h 127.0.0.1 -U postgres -c "CREATE TABLESPACE pixiebob_log_space OWNER pixiebob LOCATION '/var/log_data/tablespace';" >> ${INIT_PIXIEBOB_LOG} 2>&1
        /usr/pgsql-9.6/bin/createdb pixiebob_log -h 127.0.0.1 -U pixiebob -D pixiebob_log_space >> ${INIT_PIXIEBOB_LOG} 2>&1

        /usr/bin/psql -h 127.0.0.1 -U pixiebob -d pixiebob_log -c "CREATE TABLE alembic_version (version_num varchar(32));" >> ${INIT_PIXIEBOB_LOG} 2>&1
    fi

    systemctl start postgresql-9.6.service >> ${INIT_PIXIEBOB_LOG} 2>&1

    # TODO: Init log data in database
    export LOG_DATABASE_URL="postgresql://pixiebob@/pixiebob_log?application_name=DDD-portal-installer"
    export DATABASE_USER="pixiebob"
    export LOG_DATABASE_NAME="pixiebob_log"
    export APP_SETTINGS="config.DevelopmentConfig"
    export PYTHONPATH="/opt/TrendMicro/Pixiebob/Server/api:/opt/TrendMicro/Pixiebob/common_modules"
}

init_taxiiserver_for_migration() {
    if [ "$(psql -U pixiebob -lqt | cut -d \| -f 1 | grep -w taxiiserver)" == "" ]; then
        echo "Init taxiiserver" >> ${INIT_PIXIEBOB_LOG} 2>&1
        /usr/pgsql-9.6/bin/createuser -h 127.0.0.1 -U postgres taxiiserver >> ${INIT_PIXIEBOB_LOG} 2>&1
        /usr/bin/psql -h 127.0.0.1 -U postgres -c "ALTER USER taxiiserver CREATEDB;" >> ${INIT_PIXIEBOB_LOG} 2>&1
        /usr/pgsql-9.6/bin/createdb taxiiserver -h 127.0.0.1 -U taxiiserver >> ${INIT_PIXIEBOB_LOG} 2>&1
    fi

    has_alembic_version=`psql -U taxiiserver -c "select exists (select 1 from pg_tables where tablename = 'alembic_version');" | awk 'NR==3 {print $1}'`
    if [ "${has_alembic_version}" == "f" ]; then
        /usr/bin/psql -h 127.0.0.1 -U taxiiserver -d taxiiserver -c "CREATE TABLE alembic_version (version_num varchar(32));" >> ${INIT_PIXIEBOB_LOG} 2>&1
        has_content_blocks=`psql -U taxiiserver -c "select exists (select 1 from pg_tables where tablename = 'content_blocks');" | awk 'NR==3 {print $1}'`
        if [ "${has_content_blocks}" == "t" ]; then
            /usr/bin/psql -h 127.0.0.1 -U taxiiserver -d taxiiserver -c "INSERT INTO alembic_version (version_num) VALUES ('02321efc4962');" >> ${INIT_PIXIEBOB_LOG} 2>&1
        fi
    fi

    echo "Update TAXII server setting" >> ${INIT_PIXIEBOB_LOG} 2>&1
    cp -f "/opt/TrendMicro/Pixiebob/Server/init/config/taxii/defaults.yml" /opt/TrendMicro/Pixiebob/Server/api/config/
    USE_FQDN=`psql -A -t -U pixiebob -c "select value from tb_system_settings where key = 'local_repository.use_fqdn'"`
    if [ "${USE_FQDN}" == "1" ]; then
      NEW_FQDN=`hostname`
      sed -i "s/\[MGMT_IPV4\]/${NEW_FQDN}/g" "/opt/TrendMicro/Pixiebob/Server/api/config/defaults.yml"
    else
      MGMT_V4_ADDR=`ip -f inet -o addr show eth0 | sed -n '1p' | awk '{print $4}' | sed 's/\/.*$//g'`
      sed -i "s/\[MGMT_IPV4\]/${MGMT_V4_ADDR}/g" "/opt/TrendMicro/Pixiebob/Server/api/config/defaults.yml"
    fi
}

############################################################
##               Activate Service                         ##
############################################################
activate_pixiebob_services() {
    # Check server_mode
    echo "[`date`] Activate pixiebob services." >> ${INIT_PIXIEBOB_LOG} 2>&1
    echo "[`date`] Server mode:${SERVER_MODE}" >> ${INIT_PIXIEBOB_LOG} 2>&1

    # Activate common services
    # Portal service
    /usr/bin/cp -f ${SERVER_CONF_ROOT}systemd/pixiebob-portal.service /usr/lib/systemd/system/
    systemctl daemon-reload
    systemctl enable pixiebob-portal
    systemctl start pixiebob-portal

    # Scheduler service
    # Check port binding
    enable_port_binding=`/usr/bin/psql -U pixiebob -c "SELECT value from tb_system_settings where key='enable_port_binding'" -A -t` >> ${INIT_PIXIEBOB_LOG} 2>&1
    if [ "${enable_port_binding}" == "1" ]; then
        echo "[`date`] Enable port binding, use netns to launch scheduler service." >> ${INIT_PIXIEBOB_LOG} 2>&1
        /usr/bin/cp -f ${SERVER_CONF_ROOT}systemd/pixiebob-scheduler-sys.service /usr/lib/systemd/system/pixiebob-scheduler.service
    else
        echo "[`date`] Port binding not enabled, use normal scheduler service." >> ${INIT_PIXIEBOB_LOG} 2>&1
        /usr/bin/cp -f ${SERVER_CONF_ROOT}systemd/pixiebob-scheduler.service /usr/lib/systemd/system/
    fi

    systemctl daemon-reload
    systemctl enable pixiebob-scheduler
    systemctl start pixiebob-scheduler


    case ${SERVER_MODE} in
      "ALL")  # All-in-one
          # Agent service
          /usr/bin/cp -f ${SERVER_CONF_ROOT}systemd/pixiebob-agent.service /usr/lib/systemd/system/
          systemctl daemon-reload
          systemctl enable pixiebob-agent
          systemctl start pixiebob-agent

          # Resource service
          /usr/bin/cp -f ${SERVER_CONF_ROOT}systemd/pixiebob-resource.service /usr/lib/systemd/system/
          systemctl daemon-reload
          systemctl enable pixiebob-resource
          systemctl start pixiebob-resource

          # Log Receiver service
          echo "127.0.0.1 $HOSTNAME" | sudo tee -a /etc/hosts
          systemctl enable pixiebob-log-receiver
          systemctl start pixiebob-log-receiver

          # Nginx2 service
          /usr/bin/cp -f ${SERVER_CONF_ROOT}systemd/nginx2.service /usr/lib/systemd/system/
          systemctl daemon-reload
          systemctl enable nginx2.service
          systemctl start nginx2.service
          ;;

      "UC") # UC server only
          # Agent service
          /usr/bin/cp -f ${SERVER_CONF_ROOT}systemd/pixiebob-agent.service /usr/lib/systemd/system/
          systemctl daemon-reload
          systemctl enable pixiebob-agent
          systemctl start pixiebob-agent
          ;;

      "UCLR") # UC Local Repo only
          # Resource service
          /usr/bin/cp -f ${SERVER_CONF_ROOT}systemd/pixiebob-resource.service /usr/lib/systemd/system/
          systemctl daemon-reload
          systemctl enable pixiebob-resource
          systemctl start pixiebob-resource
          ;;

      "UCRR") # UC Remote Repo only
          # Resource service
          /usr/bin/cp -f ${SERVER_CONF_ROOT}systemd/pixiebob-resource.service /usr/lib/systemd/system/
          systemctl daemon-reload
          systemctl enable pixiebob-resource
          systemctl start pixiebob-resource
          ;;

      *) # Should not happen
          echo "[`date`] [ERROR] Invalid install type." >> ${INIT_PIXIEBOB_LOG} 2>&1
          ;;

    esac

}

############################################################
##               Install pbagent                          ##
############################################################
install_pbagent() {
    echo "[`date`] Install LR agent." >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/tar zxf ${AGENT_ROOT}pbagent-*-2.7.tgz -C /tmp
    cd /tmp/pbagent
    ./setup.sh -psqluser postgres -psqldb postgres -profile /opt/TrendMicro/Pixiebob/Server/init/config/profile/profile_uclr -path /opt/TrendMicro/Pixiebob/Agent -only_use_system_library >> ${INIT_PIXIEBOB_LOG} 2>&1

    echo "[`date`] Install result:$?" >> ${INIT_PIXIEBOB_LOG} 2>&1
    echo "[`date`] Remove pbagent installation package." >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/rm -rf /tmp/pbagent
}



############################################################
##               Migrate pbagent                          ##
############################################################
migrate_pbagent() {
    echo "[`date`] Migrate LR agent." >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/tar zxf ${AGENT_ROOT}pbagent-*-2.7.tgz -C /tmp
    cd /tmp/pbagent

    new_version=$(grep "PBAgent" ./version.info | cut -d= -f 2 | xargs)
    ori_version=$(grep "PBAgent" /opt/TrendMicro/Pixiebob/Agent/pbagent/agent/version.info | cut -d= -f 2 | xargs)
    ori_first=$(echo "$ori_version" | cut -d. -f 1)
    ori_second=$(echo "$ori_version" | cut -d. -f 2)
    ori_third=$(echo "$ori_version" | cut -d. -f 3)
    ori_fourth=$(echo "$ori_version" | cut -d. -f 4)
    new_first=$(echo "$new_version" | cut -d. -f 1)
    new_second=$(echo "$new_version" | cut -d. -f 2)
    new_third=$(echo "$new_version" | cut -d. -f 3)
    new_fourth=$(echo "$new_version" | cut -d. -f 4)

    if (python /opt/TrendMicro/Pixiebob/Agent/pbagent/cli/pbagent_cli.pyc status 2>/dev/null) | grep running; then
        python /opt/TrendMicro/Pixiebob/Agent/pbagent/cli/pbagent_cli.pyc stop
    fi

    echo "[`date`] Original version:[$ori_version], New version:[$new_version]" >> ${INIT_PIXIEBOB_LOG} 2>&1
    action='Migrate'
    if [ $ori_first -gt $new_first ]; then
        action='Rollback'
    elif [ $ori_first -lt $new_first ]; then
        action='Migrate'
    elif [ $ori_second -gt $new_second ]; then
        action='Rollback'
    elif [ $ori_second -lt $new_second ]; then
        action='Migrate'
    elif [ $ori_third -gt $new_third ]; then
        action='Rollback'
    elif [ $ori_third -lt $new_third ]; then
        action='Migrate'
    elif [ $ori_fourth -gt $new_fourth ]; then
        action='Rollback'
    elif [ $ori_fourth -lt $new_fourth ]; then
        action='Migrate'
    else
        action='None'
    fi


    if [ "$action" == 'Rollback' ]; then
        echo "[`date`] Rollback case.." >> ${INIT_PIXIEBOB_LOG} 2>&1
        ./setup.sh -psqluser postgres -psqldb postgres -profile /opt/TrendMicro/Pixiebob/Server/init/config/profile/profile_uclr -path /opt/TrendMicro/Pixiebob/Agent -only_use_system_library -rollback >> ${INIT_PIXIEBOB_LOG} 2>&1
        echo "[`date`] Rollback result:$?" >> ${INIT_PIXIEBOB_LOG} 2>&1
    elif [ "$action" == 'Migrate' ]; then
        echo "[`date`] Migration case.." >> ${INIT_PIXIEBOB_LOG} 2>&1
        ./setup.sh -psqluser postgres -psqldb postgres -profile /opt/TrendMicro/Pixiebob/Server/init/config/profile/profile_uclr -path /opt/TrendMicro/Pixiebob/Agent -only_use_system_library -migration >> ${INIT_PIXIEBOB_LOG} 2>&1
        echo "[`date`] Migrate result:$?" >> ${INIT_PIXIEBOB_LOG} 2>&1
    else
        echo "[`date`] No need to migrate/rollback"  >> ${INIT_PIXIEBOB_LOG} 2>&1
    fi

    echo "[`date`] Remove pbagent installation package." >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/rm -rf /tmp/pbagent
}


############################################################
##               Set Product Version                      ##
############################################################
set_product_version() {
    [ -d /etc/sysconfig ] || mkdir /etc/sysconfig/
    /usr/bin/cp -f ${SERVER_ROOT}init/product_env /etc/sysconfig/

    [ -d /etc/profile.d/ ] || mkdir /etc/profile.d/
    /usr/bin/cp -f ${SERVER_ROOT}init/product.sh /etc/profile.d/
}


############################################################
##               Set DDD Auth                             ##
############################################################
set_ddd_auth() {
    if [ ! -f "/opt/TrendMicro/Pixiebob/ddd_auth/login.defs.bk" ]; then
        # Backup file not exists, perform backup
        cp /etc/login.defs /opt/TrendMicro/Pixiebob/ddd_auth/login.defs.bk
        cp /etc/default/useradd /opt/TrendMicro/Pixiebob/ddd_auth/useradd.bk
        cp /etc/pam.d/login /opt/TrendMicro/Pixiebob/ddd_auth/login.bk
        cp /etc/pam.d/sshd /opt/TrendMicro/Pixiebob/ddd_auth/sshd.bk
        cp /etc/ssh/sshd_config /opt/TrendMicro/Pixiebob/ddd_auth/sshd_config.bk
    fi

    if [ -f "/opt/TrendMicro/Pixiebob/ddd_auth/login.defs" ]; then
        # Config file exists, overwrite
        mv -f /opt/TrendMicro/Pixiebob/ddd_auth/login.defs /etc/
        mv -f /opt/TrendMicro/Pixiebob/ddd_auth/useradd /etc/default/
        mv -f /opt/TrendMicro/Pixiebob/ddd_auth/login /etc/pam.d/
        mv -f /opt/TrendMicro/Pixiebob/ddd_auth/sshd /etc/pam.d/
        mv -f /opt/TrendMicro/Pixiebob/ddd_auth/sshd_config /etc/ssh/
        mv -f /opt/TrendMicro/Pixiebob/ddd_auth/pam_ddd_ssh.so /usr/lib64/security/
        mv -f /opt/TrendMicro/Pixiebob/ddd_auth/pam_ddd_login.so /usr/lib64/security/
        /usr/bin/systemctl try-restart sshd
    fi
}

############################################################
##               Setup ES Index                           ##
############################################################
setup_elasticsearch() {
    echo "Create elasticsearch data folder if need" >> ${INIT_PIXIEBOB_LOG} 2>&1
    [ -d /opt/TrendMicro/Pixiebob/Server/logs/elasticsearch ] || mkdir -p /opt/TrendMicro/Pixiebob/Server/logs/elasticsearch && chown elasticsearch:elasticsearch /opt/TrendMicro/Pixiebob/Server/logs/elasticsearch
    [ -d /var/log_data/elasticsearch ] || mkdir -p /var/log_data/elasticsearch && chown elasticsearch:elasticsearch /var/log_data/elasticsearch

    echo "Replace elasticsearch.yml: ${SERVER_CONF_ROOT}elasticsearch/elasticsearch.yml" >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/cp -f ${SERVER_CONF_ROOT}elasticsearch/elasticsearch.yml /etc/elasticsearch/elasticsearch.yml
    echo "Replace log4j2.properties: ${SERVER_CONF_ROOT}elasticsearch/log4j2.properties" >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/cp -f ${SERVER_CONF_ROOT}elasticsearch/log4j2.properties /etc/elasticsearch/log4j2.properties

    echo "Replace elasticsearch systemd file" >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/cp -f ${SERVER_CONF_ROOT}systemd/elasticsearch.service /usr/lib/systemd/system/
    systemctl daemon-reload

    echo "Enable Elasticsearch..." >> ${INIT_PIXIEBOB_LOG} 2>&1
    systemctl enable elasticsearch.service
    echo "Start Elasticsearch..." >> ${INIT_PIXIEBOB_LOG} 2>&1
    systemctl restart elasticsearch.service

    flag_repeat=15
    curl -s -XGET "http://localhost:9200" |grep -qw 'cluster_uuid'
    es_status=$?
    while [ $es_status -ne 0 ]
    do
        if [ $flag_repeat -eq 0 ]; then
            break
        fi
        let flag_repeat=flag_repeat-1
        sleep 5
        echo "Sleep 5 seconds to wait for elasticsearch" >> ${INIT_PIXIEBOB_LOG} 2>&1
        curl -s -XGET "http://localhost:9200" |grep -qw 'cluster_uuid'
        es_status=$?
    done
    if [ $es_status -eq 0 ]; then
        log_index_template_name="template_ddd_log"
        report_index_template_name="template_ddd_report"
        na_log_index_template_name="template_dddna_log"

        echo "Elasticsearch is ready" >> ${INIT_PIXIEBOB_LOG} 2>&1
        curl -s -XGET "http://localhost:9200/_cluster/settings" | grep -qw "logger"
        if [ $? -eq 0 ]; then
            echo "Elasticsearch cluster setting has been set. Skip to set logger level." >> ${INIT_PIXIEBOB_LOG} 2>&1
        else
            echo "Set cluster log level as 'ERROR' by default." >> ${INIT_PIXIEBOB_LOG} 2>&1
            curl -s -XPUT "http://localhost:9200/_cluster/settings" -d'{"persistent":{"logger._root":"ERROR"}}'
        fi

        echo "Check template update..." >> ${INIT_PIXIEBOB_LOG} 2>&1
        curl -s -XGET "http://localhost:9200/_template/template_ddd_log?filter_path=**.traffic" | grep -qw '"integer"'
        template_flag=$?
        echo "template_flag = ${template_flag}" >> ${INIT_PIXIEBOB_LOG} 2>&1

        echo "Set Elasticsearch template..." >> ${INIT_PIXIEBOB_LOG} 2>&1
        curl -s -XPUT "http://localhost:9200/_template/${log_index_template_name}" -d @${SERVER_ROOT}api/config/es_log_index_template_config.json >> ${INIT_PIXIEBOB_LOG} 2>&1
        curl -s -XPUT "http://localhost:9200/_template/${report_index_template_name}" -d @${SERVER_ROOT}api/config/es_report_index_template_config.json >> ${INIT_PIXIEBOB_LOG} 2>&1
        curl -s -XPUT "http://localhost:9200/_template/${na_log_index_template_name}" -d @${SERVER_ROOT}api/config/es_na_log_index_template_config.json >> ${INIT_PIXIEBOB_LOG} 2>&1

        if [ ${template_flag} -eq 0 ]; then
            echo "Need to forcibly sync current log index." >> ${INIT_PIXIEBOB_LOG} 2>&1
            python -m managers.index_manager --force_sync True --sync_log True --sync_report False >> ${INIT_PIXIEBOB_LOG} 2>&1
        else
            echo "No need to forcibly sync current log index." >> ${INIT_PIXIEBOB_LOG} 2>&1
        fi

    else
        echo "Elasticsearch is inactivated. Could not create index template." >> ${INIT_PIXIEBOB_LOG} 2>&1
    fi
}

############################################################
##               Setup atop config                        ##
############################################################
setup_atop() {
    echo "Create atop log folder if need" >> ${INIT_PIXIEBOB_LOG} 2>&1
    [ -d  /opt/TrendMicro/Pixiebob/Server/logs/atop ] || mkdir -p /opt/TrendMicro/Pixiebob/Server/logs/atop

    grep -qw "^LOGPATH=/opt/TrendMicro/Pixiebob/Server/logs/atop" /etc/sysconfig/atop
    if [ $? -ne 0 ]; then
        echo "Replace atop config /etc/sysconfig/atop: ${SERVER_CONF_ROOT}atop/atop.sysconfig" >> ${INIT_PIXIEBOB_LOG} 2>&1
        /usr/bin/cp -f ${SERVER_CONF_ROOT}atop/atop.sysconfig /etc/sysconfig/atop
        echo "Replace logrotate config /etc/logrotate.d/atop: ${SERVER_CONF_ROOT}atop/atop.logrotate" >> ${INIT_PIXIEBOB_LOG} 2>&1
        /usr/bin/cp -f ${SERVER_CONF_ROOT}atop/atop.logrotate /etc/logrotate.d/atop
        echo "Add cron job for atop" >> ${INIT_PIXIEBOB_LOG} 2>&1
        echo '5 0 * * * root /usr/bin/find /opt/TrendMicro/Pixiebob/Server/logs/atop -name "atop_*" -mtime +14 -exec rm {} \;' >> /etc/cron.d/atop
    fi

    # For migration, reset log keep day from 28 to 14
    sed -i "s/-mtime +28/-mtime +14/" /etc/cron.d/atop
    systemctl reload crond.service

    if [ -d /var/local_data/log/atop -a -d /opt/TrendMicro/Pixiebob/Server/logs/atop ]; then
        echo "Move /var/local_data/log/atop/atop_* to /opt/TrendMicro/Pixiebob/Server/logs/atop/" >> ${INIT_PIXIEBOB_LOG} 2>&1
        /usr/bin/cp -f /var/local_data/log/atop/atop_* /opt/TrendMicro/Pixiebob/Server/logs/atop/

        echo "Remove /var/local_data/log/atop" >> ${INIT_PIXIEBOB_LOG} 2>&1
        rm -rf /var/local_data/log/atop

        echo "Remove previous atop crond setting" >> ${INIT_PIXIEBOB_LOG} 2>&1
        sed -i "/\/var\/local_data\/log\/atop/d" /etc/cron.d/atop
    fi

    echo "Update atop service file" >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/cp -f ${SERVER_CONF_ROOT}systemd/atop.service /usr/lib/systemd/system/
    systemctl daemon-reload

    echo "Enable atop service" >> ${INIT_PIXIEBOB_LOG} 2>&1
    systemctl enable atop.service
    echo "Reload/Restart atop service" >> ${INIT_PIXIEBOB_LOG} 2>&1
    systemctl reload-or-restart atop.service
}

############################################################
##               Setup Dump Handler                       ##
############################################################
setup_dump_handler() {
    echo "Setup Dump Handler..." >> ${INIT_PIXIEBOB_LOG} 2>&1
    ulimit -c unlimited
    systemctl disable abrtd.service
    systemctl disable abrt-xorg.service
    systemctl disable abrt-vmcore.service
    systemctl disable abrt-oops.service
    systemctl disable abrt-ccpp.service
    # dump sizes 1 Gbytes
    echo "local3.* /opt/TrendMicro/Pixiebob/Server/logs/dump_handler/dump_handler.log" > /etc/rsyslog.d/dump_handler.conf
    systemctl restart rsyslog.service
    echo ${DUMP_CONFIG} >  /proc/sys/kernel/core_pattern
    cat << EOF > /etc/logrotate.d/dump_handler.conf
/opt/TrendMicro/Pixiebob/Server/logs/dump_handler {
    daily
    missingok
    rotate 4
    compress
    delaycompress
    notifempty
    size 2048k
    create 0600 root root
}
EOF

    grep -qa ${DUMP_PATTERN} /etc/sysctl.conf
    if [ $? -ne 0 ]; then
        echo "${DUMP_PATTERN}=${DUMP_CONFIG}" >> /etc/sysctl.conf
    else
        sed -i "s/${DUMP_PATTERN}=.*/${DUMP_PATTERN}=${DUMP_CONFIG_ESC}/" /etc/sysctl.conf
    fi
}

############################################################
##               Setup 3rd party integration              ##
############################################################
setup_3rd_party_int() {
    echo "Setup 3rd party integration..." >> ${INIT_PIXIEBOB_LOG} 2>&1

    if [ ! -d /etc/nginx2 ]; then
        cp -rf /etc/nginx /etc/nginx2 >> ${INIT_PIXIEBOB_LOG} 2>&1
        rm -f /etc/nginx2/*.conf >> ${INIT_PIXIEBOB_LOG} 2>&1
        rm -f /etc/nginx2/conf.d/*.conf >> ${INIT_PIXIEBOB_LOG} 2>&1
        rm -f /etc/nginx2/conf.d/ioc/*.conf >> ${INIT_PIXIEBOB_LOG} 2>&1
    fi
    mkdir -p /etc/nginx2/conf.d/ioc >> ${INIT_PIXIEBOB_LOG} 2>&1
    mkdir -p /etc/nginx/conf.d/ioc >> ${INIT_PIXIEBOB_LOG} 2>&1

    # default empty config for backup restore to "disabled"
    /usr/bin/touch /etc/nginx/conf.d/ioc/ioc_pal_https.conf >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/touch /etc/nginx/conf.d/ioc/ioc_blu_https.conf >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/touch /etc/nginx2/conf.d/ioc/ioc_pal_http.conf >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/touch /etc/nginx2/conf.d/ioc/ioc_blu_http.conf >> ${INIT_PIXIEBOB_LOG} 2>&1

    cp -f ${SERVER_CONF_ROOT}nginx2/nginx.conf /etc/nginx2 >> ${INIT_PIXIEBOB_LOG} 2>&1

    mkdir -p ${SERVER_ROOT}api/third_party_int/nginx_config >> ${INIT_PIXIEBOB_LOG} 2>&1
    cp -f ${SERVER_CONF_ROOT}nginx/ioc/*.conf ${SERVER_ROOT}api/third_party_int/nginx_config >> ${INIT_PIXIEBOB_LOG} 2>&1
    cp -f ${SERVER_CONF_ROOT}nginx2/ioc/*.conf ${SERVER_ROOT}api/third_party_int/nginx_config >> ${INIT_PIXIEBOB_LOG} 2>&1

    mkdir -p /var/log_data/nginx_ioc_root/blu >> ${INIT_PIXIEBOB_LOG} 2>&1
    mkdir -p /var/log_data/nginx_ioc_root/pal >> ${INIT_PIXIEBOB_LOG} 2>&1
    mkdir -p /var/log_data/nginx_ioc_root/generation >> ${INIT_PIXIEBOB_LOG} 2>&1
    mkdir -p /var/log/nginx2 >> ${INIT_PIXIEBOB_LOG} 2>&1
    mkdir -p /var/log_data/ioc >> ${INIT_PIXIEBOB_LOG} 2>&1

    cat << EOF > /etc/logrotate.d/nginx2
/var/log/nginx2 {
    daily
    missingok
    rotate 4
    compress
    delaycompress
    notifempty
    size 2048k
    create 0600 root root
}
EOF

}

setup_yara_int() {
    echo "Setup Yara package folder..." >> ${INIT_PIXIEBOB_LOG} 2>&1
    mkdir -p /var/local_data/yara_package >> ${INIT_PIXIEBOB_LOG} 2>&1
}

setup_taxii_int() {
    echo "Setup TAXII server package folder..." >> ${INIT_PIXIEBOB_LOG} 2>&1
    mkdir -p /var/local_data/taxii/tmp >> ${INIT_PIXIEBOB_LOG} 2>&1
    mkdir -p /var/local_data/taxii/poll_tmp >> ${INIT_PIXIEBOB_LOG} 2>&1
}

setup_stix_int() {
    echo "Setup STIX server package folder..." >> ${INIT_PIXIEBOB_LOG} 2>&1
    mkdir -p /var/local_data/stix/tmp >> ${INIT_PIXIEBOB_LOG} 2>&1
}

############################################################
##          Setup .xsd folder for STIX validator          ##
############################################################
setup_xsd_folder() {
    echo "Setup .xsd folder for STIX validator..." >> ${INIT_PIXIEBOB_LOG} 2>&1
    # For version 1.1.1
    DIR="/usr/lib/python2.7/site-packages/sdv/resources/xsd/stix_1.1.1/external/ddd"
    if [ ! -d "$DIR" ]; then
        mkdir -p ${DIR} >> ${INIT_PIXIEBOB_LOG} 2>&1
    fi
    cp -rf ${SERVER_API_DATA_ROOT}xsd/stix_1.1.1/*.xsd ${DIR}/
}


############################################################
##          Remove unused packages and services           ##
############################################################
remove_unused_packages() {

    # Check deplicated packages
    rpm -q tcpdump
    flag_remove_tcpdump=$?
    if [ ${flag_remove_tcpdump} -eq 0 ]; then
        echo "[`date`] Remove unused tcpdump package" >> ${INIT_PIXIEBOB_LOG} 2>&1
        rpm -e tcpdump
    fi

    rpm -q flash-plugin
    flag_remove_flash=$?
    if [ ${flag_remove_flash} -eq 0 ]; then
        echo "[`date`] Remove unused flash-plugin package" >> ${INIT_PIXIEBOB_LOG} 2>&1
        rpm -e flash-plugin
    fi

    # Check unused system services
    rpm -q rpcbind
    flag_rpc_installed=$?
    if [ ${flag_rpc_installed} -eq 0 ]; then
        rpc_enabled=`systemctl is-enabled rpcbind`
        if [ "${rpc_enabled}" != "masked" ]; then
            echo "[`date`] Disable rpcbind service" >> ${INIT_PIXIEBOB_LOG} 2>&1
            systemctl stop rpcbind.socket
            systemctl disable rpcbind.socket
            systemctl stop rpcbind
            systemctl disable rpcbind
            systemctl mask rpcbind                       
        fi
    fi
}


# Main logic
if [ ! -f ${INIT_PIXIEBOB_FLAG} ]; then
    # Setup dump_handler
    setup_dump_handler

    # Initialization (first installation)
    echo "[`date`] Fresh install case, initialize data." >> ${INIT_PIXIEBOB_LOG} 2>&1
    init_database
    init_data
    set_product_version
    set_ddd_auth

    # Install LR agent in UCLR mode
    if [ ${SERVER_MODE} == "UCLR" ]; then
        install_pbagent
    fi

    if [ ${SERVER_MODE} == "ALL" ]; then
        # Init log data
        init_log_data
        # Enable Elasticsearch and set template
        setup_elasticsearch

        # Cron job will be added in consolidated mode
        /usr/bin/cp -f ${SERVER_CRON_ROOT}* /etc/cron.d/
        systemctl reload crond.service
    fi

    # 3rd part integration
    setup_3rd_party_int

    # yara package folder setup
    setup_yara_int

    # TAXII server folder setup
    setup_taxii_int

    # STIX server folder setup
    setup_stix_int

    # .xsd folder setup
    setup_xsd_folder

    # Setup atop
    setup_atop
    activate_pixiebob_services

    # Remove unused packages
    remove_unused_packages

    # Set init flag
    /usr/bin/touch ${INIT_PIXIEBOB_FLAG}
elif [ -f ${MIGRATE_FLAG} -a ! -f ${OPENVA_UPGRADE_FLAG} ]; then
    # Setup dump_handler
    setup_dump_handler

    # Check if tb_ioc_data exists
    /usr/bin/psql -d pixiebob -U pixiebob -q -c "\d tb_ioc_data" 2>&1
    exist_ioc_data=$?
    echo "[`date`] Check if tb_ioc_data exists? ${exist_ioc_data}" >> ${INIT_PIXIEBOB_LOG} 2>&1

    # Migrate case, need to launch data init again
    echo "[`date`] Migrate case, re-initialize data again." >> ${INIT_PIXIEBOB_LOG} 2>&1
    init_taxiiserver_for_migration
    init_data
    set_product_version
    set_ddd_auth

    # Clear hotfix history
    echo "[`date`] Clear hotfix history." >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/psql -h 127.0.0.1 -U pixiebob -c "DELETE FROM tb_updates;" >> ${INIT_PIXIEBOB_LOG} 2>&1

    echo "[`date`] Remove openva hotfix history." >> ${INIT_PIXIEBOB_LOG} 2>&1
    echo -e "[Patches]\n" > /usr/local/openva-update-tools/patchmgm.ini
    rm -rf /var/app_data

    # Remove LEEF syslog
    echo "[`date`] Remove LEEF syslog." >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/psql -h 127.0.0.1 -U pixiebob -c "DELETE FROM tb_syslog_forward_task WHERE server_id IN (SELECT id FROM tb_syslog_forward_server WHERE format = 1); DELETE FROM tb_syslog_forward_server WHERE format = 1;" >> ${INIT_PIXIEBOB_LOG} 2>&1

    # Install LR agent in UCLR mode
    if [ ${SERVER_MODE} == "UCLR" ]; then
        migrate_pbagent
    fi

    # Migrate textUI accounts
    if [ -f "${TEXTUI_ACCOUNTS}" ]; then
        echo "[`date`] Migrate textUI accounts from previous version." >> ${INIT_PIXIEBOB_LOG} 2>&1
        local acc_pwd_list=$(cat "${TEXTUI_ACCOUNTS}")
        for acc_pwd in ${acc_pwd_list}
        do
            account=$(echo "$acc_pwd" | awk -F":" '{print $1}')
            ret=$(cat /etc/passwd | awk '{FS=":"} $1 == name {print $1}' name="${account}" )
            if [ "${ret}" == "${account}" ]; then
                echo "[`date`] User [${account}] exists, delete it." >> ${INIT_PIXIEBOB_LOG} 2>&1
                /usr/sbin/userdel --remove "${account}" 2>&1 > /dev/null
            fi
            /usr/sbin/useradd "${account}" 2>&1 > /dev/null
            chsh -s "/opt/TrendMicro/Pixiebob/textUI/admin_shell" "${account}" 2>&1 > /dev/null
            echo "${acc_pwd}" | chpasswd -e 2>&1 > /dev/null
        done
        rm -f "${TEXTUI_ACCOUNTS}"
    fi

    if [ ${SERVER_MODE} == "ALL" ]; then
        # Init log data
        init_log_data
        # Enable Elasticsearch and set template
        setup_elasticsearch

        # Cron job will be added in consolidated mode
        /usr/bin/cp -f ${SERVER_CRON_ROOT}* /etc/cron.d/
        systemctl reload crond.service
    fi

    # If exist_ioc_data != 0, port ioc_data from ES to DB
    if [ ${exist_ioc_data} -ne 0 ]; then
        echo "[`date`] Try to port IOC data from ES to DB" >> ${INIT_PIXIEBOB_LOG} 2>&1
        python -m utils.command_helper --module managers.ioc_data_manager --function port_es_to_db --logger pixiebob-api --log_level DEBUG >> ${INIT_PIXIEBOB_LOG} 2>&1
    else
        echo "[`date`] No need to port IOC data from ES to DB" >> ${INIT_PIXIEBOB_LOG} 2>&1
    fi

    # Reload nginx
    reload_nginx_conf

    # 3rd part integration
    setup_3rd_party_int

    # yara package folder setup
    setup_yara_int

    # TAXII server folder setup
    setup_taxii_int

    # STIX server folder setup
    setup_stix_int

    # .xsd folder setup
    setup_xsd_folder

    # Setup atop
    setup_atop
    activate_pixiebob_services

    # Remove unused packages
    remove_unused_packages

    echo "[`date`] Remove migrate flag." >> ${INIT_PIXIEBOB_LOG} 2>&1
    /usr/bin/rm -f "${MIGRATE_FLAG}"
else
    echo "[`date`] Pixiebob data already initialized, skip." >> ${INIT_PIXIEBOB_LOG} 2>&1
fi

# Reset service stopped alert by removing service status file
echo "[`date`] Reset service stopped alert." >> ${INIT_PIXIEBOB_LOG} 2>&1
/usr/bin/rm -f ${SERVER_API_DATA_ROOT}ga_service_status.ini

# Check server api_key, re-generate if api_key is empty
systemctl start postgresql-9.6.service  2>&1
api_key=`/usr/bin/psql -U pixiebob -c "SELECT value from tb_system_settings where key='api_key'" -A -t` >> ${INIT_PIXIEBOB_LOG} 2>&1
if [ $? -eq 0 ]; then
    if [ "${api_key}" == "" ]; then
        echo "[`date`] The api_key of DDD server has been cleared, re-generate it." >> ${INIT_PIXIEBOB_LOG} 2>&1
        uuid=$(uuidgen)
        new_api_key=${uuid^^}
        modified_time=`date`
        result=`/usr/bin/psql -U pixiebob -c "Update tb_system_settings set value='${new_api_key}', modified_time='${modified_time}' where key='api_key'"` >> ${INIT_PIXIEBOB_LOG} 2>&1
        if [ $? -ne 0 ]; then
            echo "[`date`] Fail to update API key, skip." >> ${INIT_PIXIEBOB_LOG} 2>&1
        fi
    fi
else
    echo "[`date`] Fail to retrieve API key, skip." >> ${INIT_PIXIEBOB_LOG} 2>&1
fi

wsi_api_key=`/usr/bin/psql -U pixiebob -c "SELECT value from tb_system_settings where key='wsi.api_key'" -A -t` >> ${INIT_PIXIEBOB_LOG} 2>&1
if [ $? -eq 0 ]; then
    if [ "${wsi_api_key}" == "" ]; then
        echo "[`date`] The wsi_api_key of DDD server has been cleared, re-generate it." >> ${INIT_PIXIEBOB_LOG} 2>&1
        uuid=$(uuidgen)
        new_api_key=${uuid^^}
        modified_time=`date`
        result=`/usr/bin/psql -U pixiebob -c "Update tb_system_settings set value='${new_api_key}', modified_time='${modified_time}' where key='wsi.api_key'"` >> ${INIT_PIXIEBOB_LOG} 2>&1
        if [ $? -ne 0 ]; then
            echo "[`date`] Fail to update WSI API key, skip." >> ${INIT_PIXIEBOB_LOG} 2>&1
        fi
    fi
else
    echo "[`date`] Fail to retrieve WSI API key, skip." >> ${INIT_PIXIEBOB_LOG} 2>&1
fi


# Write flag file to indicate system has booted
echo ${SERVER_MODE} >> /tmp/system_boot_checker.flag

# Remove open-va unpack folder if exists
/usr/bin/rm -rf /opt/TrendMicro/openva-update-tools
